using System;
using System.Data;
using System.Reflection;
using Common = gov.va.med.vbecs.Common;
using gov.va.med.vbecs.DAL;
using gov.va.med.vbecs.ExceptionManagement;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;

namespace gov.va.med.vbecs.BOL
{
		#region Header

    ///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
    ///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
    ///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
    ///<Developers>
    ///	<Developer>Cameron Taylor</Developer>
    ///	<Developer>Mike Smith</Developer>
    ///	<Developer>Luke Meyer</Developer>
    ///	<Developer>Carl Jensen</Developer>
    ///</Developers>
    ///<SiteName>Hines OIFO</SiteName>
    ///<CreationDate>9/17/2002</CreationDate>
    ///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
    ///<summary>
    ///IncomingShipment business layer object
    ///</summary>

		#endregion

    public class IncomingShipment : BaseShipment
    {

		#region Variables

        private BOL.LocalSupplier _localSupplier;
		private Common.ShipmentType _shipmentType;
        private Common.ShipmentDirection _shipmentDirection;
		private BOL.ProductShipper _productShipper;

		#endregion

		#region Change Events

        /// <summary>
        /// IncomingShipment loaded event handler
        /// </summary>
        public event EventHandler ShipmentLoaded;
        /// <summary>
        /// IncomingShipment cleared event handler
        /// </summary>
        public event EventHandler ShipmentCleared;

        /// <summary>
        /// IncomingShipment loaded event
        /// </summary>
        protected virtual void OnShipmentLoaded()
        {
            if (ShipmentLoaded != null)
                ShipmentLoaded(this, EventArgs.Empty);
        }

        /// <summary>
        /// IncomingShipment changed event
        /// </summary>
        protected virtual void OnShipmentChanged()
        {
            if(ShipmentCleared != null)
                ShipmentCleared(this, EventArgs.Empty);
        }

        
       
		#endregion

		#region Constructors


		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/21/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="253"> 
		///		<ExpectedInput>Invoice Number, Localsupplier Guid</ExpectedInput>
		///		<ExpectedOutput>Arraylist</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="265"> 
		///		<ExpectedInput>Empty InvoiceNumber</ExpectedInput>
		///		<ExpectedOutput>ArgumentNull Exception</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetIncomingShipments
		/// </summary>
		/// <param name="invoiceNumber">invoiceNumber</param>
		/// <param name="localSupplierGUID">localSupplierGUID</param>
		/// <returns></returns>
		public static System.Collections.ArrayList GetIncomingShipments(string invoiceNumber,
			System.Guid localSupplierGUID)
		{
	
			if( invoiceNumber == null )
				throw( new ArgumentNullException( "invoiceNumber" ) );

		
			
			System.Collections.ArrayList incomingShipments = new System.Collections.ArrayList();
		
			DataTable dt = DAL.Shipment.GetIncomingShipment(invoiceNumber, localSupplierGUID);
			
			if(dt.Rows.Count == 1)
			{	//IncomingShipment exists
				BOL.IncomingShipment incomingShipment = new BOL.IncomingShipment(dt.Rows[0]);
				incomingShipments.Add(incomingShipment);
			}
			else if(dt.Rows.Count > 1)
			{
				foreach(System.Data.DataRow dtRow in dt.Rows)
				{
					BOL.IncomingShipment incomingShipment = new BOL.IncomingShipment(dtRow);
					incomingShipments.Add(incomingShipment);
				}
			}

			return incomingShipments;	
		}
		
		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/21/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="266"> 
		///		<ExpectedInput>Invoice Number, Localsupplier Guid, Date</ExpectedInput>
		///		<ExpectedOutput>Arraylist</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="267"> 
		///		<ExpectedInput>Empty InvoiceNumber</ExpectedInput>
		///		<ExpectedOutput>ArgumentNull Exception</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetIncomingShipments
		/// </summary>
		/// <param name="invoiceNumber">invoiceNumber</param>
		/// <param name="localSupplierGUID">localSupplierGUID</param>
		/// <param name="processingDate">processingDate</param>
		/// <returns></returns>
		public static System.Collections.ArrayList GetIncomingShipments(string invoiceNumber,
			System.Guid localSupplierGUID, System.DateTime processingDate)
		{

			if( invoiceNumber == null )
				throw( new ArgumentNullException( "invoiceNumber" ) );
		

			System.Collections.ArrayList incomingShipments = new System.Collections.ArrayList();
			if (( localSupplierGUID != Guid.Empty ) && 
				( invoiceNumber != string.Empty ) )
			{				
				DataTable dt = DAL.Shipment.GetIncomingShipment(invoiceNumber, localSupplierGUID, processingDate);

				if(dt.Rows.Count == 1)
				{	//IncomingShipment exists
					BOL.IncomingShipment incomingShipment = new BOL.IncomingShipment(dt.Rows[0]);
					incomingShipments.Add(incomingShipment);
				}
				else if(dt.Rows.Count > 1)
				{
					foreach(System.Data.DataRow dtRow in dt.Rows)
					{
						BOL.IncomingShipment incomingShipment = new BOL.IncomingShipment(dtRow);
						incomingShipments.Add(incomingShipment);
					}
				}
			}
			return incomingShipments;	
		}
		
		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/21/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="271"> 
		///		<ExpectedInput>Invoice Number, Localsupplier Guid, Date</ExpectedInput>
		///		<ExpectedOutput>IncomingShipment object</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="272"> 
		///		<ExpectedInput>Empty InvoiceNumber</ExpectedInput>
		///		<ExpectedOutput>ArgumentNull Exception</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// IncomingShipment
		/// </summary>
		/// <param name="invoiceNumber">invoiceNumber</param>
		/// <param name="localSupplierGUID">localSupplierGUID</param>
		/// <param name="productShipperGuid">productShipperGuid</param>
		/// <param name="processingDate">processingDate</param>
		public IncomingShipment(string invoiceNumber, System.Guid localSupplierGUID, System.Guid productShipperGuid, System.DateTime processingDate)
		{
			if( invoiceNumber == null )
				throw( new ArgumentNullException( "invoiceNumber" ) );
			

            if (( localSupplierGUID != Guid.Empty ) && 
				( invoiceNumber != string.Empty ) )
			{
				DataTable dt = DAL.Shipment.GetIncomingShipment(invoiceNumber, localSupplierGUID, processingDate);

				if (dt.Rows.Count == 0)													
				{	//IncomingShipment doesn't exist
					this.ShipmentGuid = System.Guid.Empty;
					this.InvoiceNumber = invoiceNumber;
					if(this.LocalSupplier != null)
					{
						this.LocalSupplier.LocalSupplierGuid = localSupplierGUID;
					}
					else
					{
						this.LocalSupplier = new BOL.LocalSupplier(localSupplierGUID, true);
					}
					if(this.ProductShipper != null)
					{
						this.ProductShipper.ProductShipperGuid = productShipperGuid;
					}
					else
					{
						this.ProductShipper = new ProductShipper(productShipperGuid);
					}
				}							
				else
				{	//IncomingShipment exists
					this.LoadFromDataRow(dt.Rows[0]);
				}
				
				this.ShipmentDirection = Common.ShipmentDirection.IncomingShipment;
			}
		}



        ///<Developers>
        ///	<Developer>Krzysztof Dobranowski</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>2/26/2003</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="2421"> 
        ///		<ExpectedInput>DataRow containing IncomingShipment data</ExpectedInput>
        ///		<ExpectedOutput>IncomingShipment object</ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="1" testid ="2445"> 
        ///		<ExpectedInput>NA</ExpectedInput>
        ///		<ExpectedOutput>NA</ExpectedOutput>
        ///	</Case>
        ///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// Create IncomingShipment object and populate it with data from DataRow
        /// </summary>
        /// <param name="dr"></param>
        public IncomingShipment(DataRow dr)
        {
			this.ShipmentDirection = Common.ShipmentDirection.IncomingShipment;
            this.LoadFromDataRow(dr);
        }
        

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/26/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="273"> 
		///		<ExpectedInput>Shpiment Guid</ExpectedInput>
		///		<ExpectedOutput>IncomingShipment object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="274"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
        /// <summary>
        /// Constructor
        /// </summary>
        /// <param name="shipmentGuid"></param>
        public IncomingShipment(System.Guid shipmentGuid) : base(shipmentGuid)
        {
			this.ShipmentDirection = Common.ShipmentDirection.IncomingShipment;
            if (this.LocalSupplier == null) 
            {
                this.LocalSupplier = new BOL.LocalSupplier(this.LocalSupplierGuid, true);
            }
			if (this.ProductShipper == null) 
			{
				this.ProductShipper = new BOL.ProductShipper(this.ProductShipperGuid);
			}
        }

		#endregion

		#region Properties
        
		///<Developers>
        ///	<Developer>Luke Meyer</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>1/21/2003</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="1332"> 
        ///		<ExpectedInput>Valid local supplier</ExpectedInput>
        ///		<ExpectedOutput>Valid local supplier</ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="1" testid ="1523"> 
        ///		<ExpectedInput>NA</ExpectedInput>
        ///		<ExpectedOutput>NA</ExpectedOutput>
        ///	</Case>
        ///
        ///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// Local Supplier for shipment
        /// </summary>
        public BOL.LocalSupplier LocalSupplier
        {
            get
            {
                return this._localSupplier;
            }
            set
            {
                this._localSupplier = value;
            }
        }

		///<Developers>
		///	<Developer>Software Architectects Inc.</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/4/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6825"> 
		///		<ExpectedInput>ProductShipper</ExpectedInput>
		///		<ExpectedOutput>ProductShipper</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6826"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// ProductShipper
		/// </summary>
		public BOL.ProductShipper ProductShipper
		{
			get
			{
				return this._productShipper;
			}
			set
			{
				this._productShipper = value;
			}
		}
        ///<Developers>
        ///	<Developer>Luke Meyer</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>1/21/2003</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="1333"> 
        ///		<ExpectedInput>Valid shipment type</ExpectedInput>
        ///		<ExpectedOutput>No shipment type broken rule</ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="1" testid ="1528"> 
        ///		<ExpectedInput>Unknown shipment type</ExpectedInput>
        ///		<ExpectedOutput>IncomingShipment type broken rule</ExpectedOutput>
        ///	</Case>
        ///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// Type of shipment
        /// </summary>
        public Common.ShipmentType ShipmentType
        {
            get
            {
                return this._shipmentType;
            }
            set
            {
                this._shipmentType = value;
                RuleBroken(MethodBase.GetCurrentMethod().Name,(this._shipmentType==Common.ShipmentType.Unknown));
            }
        }

        ///<Developers>
        ///	<Developer>Luke Meyer</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>1/21/2003</CreationDate>
        ///<TestCases>
        ///
        ///<Case type="0" testid ="1334"> 
        ///		<ExpectedInput>Empty shipment GUID</ExpectedInput>
        ///		<ExpectedOutput>IncomingShipment GUID is equal to an empty GUID</ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="1" testid ="1524"> 
        ///		<ExpectedInput>NA</ExpectedInput>
        ///		<ExpectedOutput>NA</ExpectedOutput>
        ///	</Case>
        ///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// IncomingShipment GUID
        /// </summary>
        public new Guid ShipmentGuid
        {
            get
            {
                return base.ShipmentGuid;
            }
            set
            {
                base.ShipmentGuid = value;
            }
        }

        ///<Developers>
        ///	<Developer>Luke Meyer</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>1/21/2003</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="1335"> 
        ///		<ExpectedInput>Valid shipment direction</ExpectedInput>
        ///		<ExpectedOutput>No shipment direction broken rule</ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="1" testid ="1527"> 
        ///		<ExpectedInput>Unknown shipment direction</ExpectedInput>
        ///		<ExpectedOutput>IncomingShipment direction broken rule</ExpectedOutput>
        ///	</Case>
        ///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// IncomingShipment direction
        /// </summary>
        public new Common.ShipmentDirection ShipmentDirection
        {
            get
            {
                return _shipmentDirection;
            }

            set
            {
                _shipmentDirection = value;
                RuleBroken(MethodBase.GetCurrentMethod().Name,(_shipmentDirection==Common.ShipmentDirection.Unknown));
            }
        }

        ///<Developers>
        ///	<Developer>Luke Meyer</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>1/21/2003</CreationDate>
        ///<TestCases>
        ///
        ///<Case type="0" testid ="1336"> 
        ///		<ExpectedInput>Valid invoice number</ExpectedInput>
        ///		<ExpectedOutput>No additional broken rules</ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="1" testid ="1525"> 
        ///		<ExpectedInput>Invalid invoice number</ExpectedInput>
        ///		<ExpectedOutput>An additional broken rule</ExpectedOutput>
        ///	</Case>
        ///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// Invoice number
        /// </summary>
        public new string InvoiceNumber
        {
            get
            {
                return base.InvoiceNumber;
            }
            set
            {
                base.InvoiceNumber = value;
                RuleBroken(MethodBase.GetCurrentMethod().Name,(base.InvoiceNumber == string.Empty));
            }
        }

        ///<Developers>
        ///	<Developer>Luke Meyer</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>1/21/2003</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="1337"> 
        ///		<ExpectedInput>Valid date</ExpectedInput>
        ///		<ExpectedOutput>No additional broken rules</ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="1" testid ="1526"> 
        ///		<ExpectedInput>Invalid date</ExpectedInput>
        ///		<ExpectedOutput>An additional broken rules</ExpectedOutput>
        ///	</Case>
        ///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// IncomingShipment date
        /// </summary>
        public new DateTime ShipmentDate
        {
            get
            {
                return base.ShipmentDate;
            }
            set
            {
                if (this.ShipmentDirection == Common.ShipmentDirection.IncomingShipment)
                {
					bool bBroken = (value.Date > VBECSDateTime.GetDivisionCurrentDateTime().Date);
                    RuleBroken(MethodBase.GetCurrentMethod().Name, bBroken);
					if (bBroken) return;
                }
				base.ShipmentDate = value;
            }
        }

		
		#endregion

		#region Methods

          
        ///<Developers>
        ///	<Developer>Mike Smith</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>2/20/2003</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="2209"> 
        ///		<ExpectedInput>NA</ExpectedInput>
        ///		<ExpectedOutput>DataTable</ExpectedOutput>
        ///	</Case>
		///	
		///<Case type="1" testid ="252"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// Builds an empty DataTable
        /// </summary>
        public static System.Data.DataTable GetEmptyShipmentSaveTable()
        {
            System.Data.DataTable dtShipments = new System.Data.DataTable();

            dtShipments.Columns.Add(TABLE.Shipment.ShipmentGuid, typeof(System.Guid));
            dtShipments.Columns.Add(TABLE.Shipment.LocalSupplierGuid, typeof(System.Guid));
            dtShipments.Columns.Add(TABLE.Shipment.InvoiceNumber, typeof(String));
            dtShipments.Columns.Add(TABLE.Shipment.ShipmentDate, typeof(DateTime));
            dtShipments.Columns.Add(TABLE.Shipment.ShipmentTypeCode, typeof(char));
            dtShipments.Columns.Add(TABLE.Shipment.ShipmentDirection, typeof(bool));
			
            return dtShipments;
        }

        ///<Developers>
        ///	<Developer>Luke Meyer</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>1/14/2003</CreationDate>
        /// <summary>
        /// Loads an object from a data row
        /// </summary>
        protected override void LoadFromDataRow(System.Data.DataRow dtRow)
        {
			this.ShipmentDirection = Common.ShipmentDirection.IncomingShipment;
            this.ShipmentDate = (DateTime) dtRow[TABLE.Shipment.ShipmentDate];
            this.ShipmentGuid = new Guid(dtRow[TABLE.Shipment.ShipmentGuid].ToString());
		
			if(this.LocalSupplier == null)
			{
				if(dtRow.Table.Columns.Contains(TABLE.Shipment.LocalSupplierGuid))
				{
					if(dtRow[TABLE.Shipment.LocalSupplierGuid].ToString().Length > 0)
					{
						this.LocalSupplier = new BOL.LocalSupplier((System.Guid) dtRow[TABLE.Shipment.LocalSupplierGuid],true);
						this.LocalSupplierGuid = this.LocalSupplier.LocalSupplierGuid;
					}
				}
			}
			else
			{
				if(dtRow.Table.Columns.Contains(TABLE.Shipment.LocalSupplierGuid))
				{
					if(dtRow[TABLE.Shipment.LocalSupplierGuid].ToString().Length > 0)
					{
						if(((System.Guid) dtRow[TABLE.Shipment.LocalSupplierGuid]) != this.LocalSupplier.LocalSupplierGuid)
						{
							this.LocalSupplier = new BOL.LocalSupplier((System.Guid) dtRow[TABLE.Shipment.LocalSupplierGuid],true);
							this.LocalSupplierGuid = this.LocalSupplier.LocalSupplierGuid;
						}
					}
				}
			}

			if(this.ProductShipper == null)
			{
				if(dtRow.Table.Columns.Contains(TABLE.Shipment.ProductShipperGuid))
				{
					if(dtRow[TABLE.Shipment.ProductShipperGuid].ToString().Length > 0)
					{
						this.ProductShipper = new BOL.ProductShipper((System.Guid) dtRow[TABLE.Shipment.ProductShipperGuid]);
						this.ProductShipperGuid = this.ProductShipper.ProductShipperGuid;
					}
				}
			}
			else
			{
				if(dtRow.Table.Columns.Contains(TABLE.Shipment.ProductShipperGuid))
				{
					if(dtRow[TABLE.Shipment.ProductShipperGuid].ToString().Length > 0)
					{
						if(((System.Guid) dtRow[TABLE.Shipment.ProductShipperGuid]) != this.ProductShipper.ProductShipperGuid)
						{
							this.ProductShipper = new BOL.ProductShipper((System.Guid) dtRow[TABLE.Shipment.ProductShipperGuid]);
							this.ProductShipperGuid = this.ProductShipper.ProductShipperGuid;
						}
					}
				}
			}
			
			if(dtRow.Table.Columns.Contains(TABLE.Shipment.InvoiceNumber))
			{
				this.InvoiceNumber = dtRow[TABLE.Shipment.InvoiceNumber].ToString();
			}
			
			if(dtRow.Table.Columns.Contains(TABLE.Shipment.ShipmentDate))
			{
				this.ShipmentDate = (System.DateTime) dtRow[TABLE.Shipment.ShipmentDate];
			}
            

        }

        ///<Developers>
        ///	<Developer>Luke Meyer</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>1/14/2003</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="1330"> 
        ///		<ExpectedInput>DataRow</ExpectedInput>
        ///		<ExpectedOutput>DataRow</ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="1" testid ="1530"> 
        ///		<ExpectedInput>NA</ExpectedInput>
        ///		<ExpectedOutput>NA</ExpectedOutput>
        ///	</Case>
        ///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        ///  Loads a data row from an object
        /// </summary>
        public override System.Data.DataRow LoadDataRowFromThis(System.Data.DataRow dtRow)
        {
            return dtRow;
        }
	

        ///<Developers>
        ///	<Developer>Luke Meyer</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>1/28/2003</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="1533"> 
        ///		<ExpectedInput>NA</ExpectedInput>
        ///		<ExpectedOutput>DataTable</ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="1" testid ="1534"> 
        ///		<ExpectedInput>NA</ExpectedInput>
        ///		<ExpectedOutput>NA</ExpectedOutput>
        ///	</Case>
        ///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// Returns a DataTable of shipments with available units from the past 5 days
        /// </summary>
        /// <returns></returns>
        public static DataTable GetShipmentsWithAvailableUnitsPastFiveDays()
        {
	        return(DAL.Shipment.GetShipmentsWithAvailableUnitsPastFiveDays());
        }

        ///<Developers>
        ///	<Developer>Luke Meyer</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>1/28/2003</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="1537"> 
        ///		<ExpectedInput>Valid Guid</ExpectedInput>
        ///		<ExpectedOutput>Data table of shipment information</ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="1" testid ="1538"> 
        ///		<ExpectedInput>Invalid Guid</ExpectedInput>
        ///		<ExpectedOutput>Empty data table</ExpectedOutput>
        ///	</Case>
        ///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// Returns a shipment based on the IncomingShipment GUID
        /// </summary>
        /// <param name="shipmentGUID"></param>
        /// <returns></returns>
        public static DataTable GetShipment(System.Guid shipmentGUID)
        {
            DataTable dtShipment = DAL.Shipment.GetShipment(shipmentGUID);
            return dtShipment;
        }

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/28/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="275"> 
		///		<ExpectedInput>Valid Guid</ExpectedInput>
		///		<ExpectedOutput>Data table of shipment information</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="282"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>Empty data table</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// //implements BR_1.40, BR_12.06
		/// </summary>
		/// <param name="bloodUnitGUID"></param>
		/// <returns></returns>
		public static DataTable GetShipmentByUnit(System.Guid bloodUnitGUID)
		{
			DataTable dtShipment = DAL.Shipment.GetShipmentByUnit(bloodUnitGUID);
			return dtShipment;
		}
		
		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/28/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="283"> 
		///		<ExpectedInput>Valid Guid</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="289"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetShipmentType
		/// </summary>
		/// <param name="shipmentGUID">shipmentGUID</param>
		/// <returns></returns>
		public static string GetShipmentType(System.Guid shipmentGUID)
		{
			DataTable dtShipment = BOL.IncomingShipment.GetShipment(shipmentGUID);
			if(dtShipment.Rows.Count > 0)
			{
				DataRow dr = dtShipment.Rows[0];
				return dr.IsNull(TABLE.Shipment.ShipmentTypeCode) ? string.Empty : dr[TABLE.Shipment.ShipmentTypeCode].ToString();
			}
			else
			{
				throw new BOL.BusinessObjectException("No shipment type found for GUID " + shipmentGUID.ToString());
			}
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/28/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="290"> 
		///		<ExpectedInput>Valid Guid</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="296"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>Empty string</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetShipmentTypeByUnit implements BR_1.40, BR_12.06
		/// </summary>
		/// <param name="bloodUnitGUID">bloodUnitGUID</param>
		/// <returns></returns>
		public static string GetShipmentTypeByUnit(System.Guid bloodUnitGUID)
		{
			DataTable dtShipment = BOL.IncomingShipment.GetShipmentByUnit(bloodUnitGUID);
			if(dtShipment.Rows.Count > 0)
			{
				DataRow dr = dtShipment.Rows[0];
				return dr.IsNull(TABLE.Shipment.ShipmentTypeCode) ? string.Empty : dr[TABLE.Shipment.ShipmentTypeCode].ToString();
			}
			else
			{
				return string.Empty;
			}
		}
		
		

	    ///<Developers>
        ///	<Developer>Mike Smith</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>1/16/2003</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="1224"> 
        ///		<ExpectedInput>blood unit, blood unit antigens, and blood unit tests.</ExpectedInput>
        ///		<ExpectedOutput>NA</ExpectedOutput>
        ///	</Case>
        ///	
        ///<Case type="1" testid ="1532"> 
        ///		<ExpectedInput>Invalid blood unit</ExpectedInput>
        ///		<ExpectedOutput>Business object exception thrown</ExpectedOutput>
        ///	</Case>
        ///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        ///  Calls the data access layer method to save an incoming shipment. BR_1.61
        /// </summary>
        /// <param name="bloodUnit">BOL.BloodUnit</param>
        /// <param name="bloodUnitAntigens">arraylist</param>
        /// <param name="bloodUnitTests">arraylist</param>
        /// <param name="lastUpdateFunctionId">UC or calling method</param>
        /// <param name="reenteredUnit">Indicates if the user is re-entering a previously dispositioned unit</param>
        /// <param name="workloadProcessID">Indicates what VBECS process should be tied to the workload event</param>
        /// <param name="listViewData">Data shown in list view for a newly added unit</param>
        public void SaveIncomingShipment(BOL.BloodUnit bloodUnit, bool reenteredUnit, System.Collections.ArrayList bloodUnitAntigens,System.Collections.ArrayList bloodUnitTests, Common.UpdateFunction lastUpdateFunctionId, string listViewData, Common.WorkloadProcessID workloadProcessID)
        {
			System.Guid [] bloodUnitGuids = new System.Guid[1];
			bloodUnitGuids[0] = bloodUnit.BloodUnitGuid;
			System.Data.DataTable dtWorkloadEvents = BOL.WorkloadEvent.GenerateWorkloadData(workloadProcessID, bloodUnitGuids, Common.WorkloadTransactionType.Unit, false);
			System.Data.DataTable dtBloodUnitAntigens = BOL.BloodUnitAntigen.GetEmptyBloodUnitAntigenTable(false);
			DataTable dtExceptionExpiredUnitReceived = null;

			foreach(BOL.BloodUnitAntigen antigen in bloodUnitAntigens)
			{
				if (antigen.RecordStatusCode == Common.RecordStatusCode.Inactive)
				{
					System.Data.DataRow dtRow = antigen.LoadDataRowFromThis(dtBloodUnitAntigens.NewRow());
					dtBloodUnitAntigens.Rows.Add(dtRow);
				}
			}
			// Get other rows (active)
			foreach(BOL.BloodUnitAntigen antigen in bloodUnitAntigens)
			{
				System.Data.DataRow dtRow = antigen.LoadDataRowFromThis(dtBloodUnitAntigens.NewRow());
				dtBloodUnitAntigens.Rows.Add(dtRow);
			}
			System.Data.DataTable dtBloodUnitTests = DAL.BloodUnitTest.GetEmptyBloodUnitTestTableSchema(false);
			foreach(Object bloodTest in bloodUnitTests)
			{
				BOL.BloodUnitTest bloodUnitTest = (BOL.BloodUnitTest) bloodTest;
				System.Data.DataRow dtRow = bloodUnitTest.LoadDataRowFromThis(dtBloodUnitTests.NewRow());
				dtBloodUnitTests.Rows.Add(dtRow);
			}
			
			//Create the BloodUnit Tables for saving
			System.Data.DataTable dtBloodUnit = DAL.BloodUnit.GetEmptyBloodUnitTableSchema(reenteredUnit);
			System.Data.DataTable dtBloodUnitMedia = DAL.BloodUnitMedia.GetEmptyBloodUnitMediaTableSchema(reenteredUnit);
			System.Data.DataTable dtBloodUnitFinance = DAL.BloodUnitFinance.GetEmptyBloodUnitFinanceTableSchema(reenteredUnit);
			System.Data.DataTable dtBloodUnitStatus = DAL.BloodUnitStatus.GetEmptyBloodUnitStatusTableSchema(reenteredUnit);

			dtBloodUnit.Rows.Add(bloodUnit.LoadDataRowFromThis(dtBloodUnit.NewRow()));
			dtBloodUnitFinance.Rows.Add(bloodUnit.BloodUnitFinance.LoadDataRowFromThis(dtBloodUnitFinance.NewRow()));
			dtBloodUnitMedia.Rows.Add(bloodUnit.BloodUnitMedia.LoadDataRowFromThis(dtBloodUnitMedia.NewRow()));
			dtBloodUnitStatus.Rows.Add(bloodUnit.BloodUnitStatus.LoadDataRowFromThis(dtBloodUnitStatus.NewRow()));

			//Create new Shipment table 
			System.Collections.ArrayList alShipmentTables =  InitializeShipmentTable(false);
			
			DataTable dtShipment = (System.Data.DataTable) alShipmentTables[0];

			DataTable dtBloodUnitShipment = (System.Data.DataTable) alShipmentTables[1];

			//Create a new Row in Shipment table and add data based on the information passed in
			DataRow drShipment = dtShipment.NewRow();
			DataRow drBloodUnitShipment = dtBloodUnitShipment.NewRow();

			//Populate the bloodunitshipment table
			drBloodUnitShipment[TABLE.BloodUnitShipment.BloodUnitShipmentGuid] = Guid.NewGuid();
			drBloodUnitShipment[TABLE.BloodUnitShipment.BloodUnitGuid] = bloodUnit.BloodUnitGuid;


			if (this.ShipmentGuid == Guid.Empty)
			{
				drBloodUnitShipment[TABLE.BloodUnitShipment.IncomingShipmentGuid] = Guid.NewGuid();
				drShipment[TABLE.Shipment.ShipmentGuid] = (System.Guid) drBloodUnitShipment[TABLE.BloodUnitShipment.IncomingShipmentGuid];
			}
			else
			{
				drBloodUnitShipment[TABLE.BloodUnitShipment.IncomingShipmentGuid] = this.ShipmentGuid;
				drShipment[TABLE.Shipment.ShipmentGuid] = this.ShipmentGuid;
			}
			drBloodUnitShipment[TABLE.BloodUnitShipment.LoginBloodProductGuid] = bloodUnit.BloodProductGuid;
			drBloodUnitShipment[TABLE.BloodUnitShipment.LoginFullProductCode] = bloodUnit.FullProductCode;
			drBloodUnitShipment[TABLE.BloodUnitShipment.LoginValues] = listViewData;
			


			drBloodUnitShipment[TABLE.BloodUnitShipment.LastUpdateUser] = Common.LogonUser.LogonUserName;
			drShipment[TABLE.Shipment.ShipmentDirection] = Convert.ToInt16(this.ShipmentDirection);
			if (this.ShipmentDate == DateTime.MinValue)
			{
				drShipment[TABLE.Shipment.ShipmentDate] = DBNull.Value;
			}
			else
			{
				drShipment[TABLE.Shipment.ShipmentDate] = this.ShipmentDate;
			}
			drShipment[TABLE.Shipment.InvoiceNumber] = this.InvoiceNumber;
			drShipment[TABLE.Shipment.LocalSupplierGuid] = this.LocalSupplier.LocalSupplierGuid;
			drShipment[TABLE.Shipment.ProductShipperGuid] = this.ProductShipper.ProductShipperGuid;
			drShipment[TABLE.Shipment.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
			drShipment[TABLE.Shipment.LastUpdateUser] = Common.LogonUser.LogonUserName;

			//Add Shipment row to the Shipment table
			dtShipment.Rows.Add(drShipment);
			dtBloodUnitShipment.Rows.Add(drBloodUnitShipment);

			//Build data set with all the tables that need to be updated/inserted
			DataSet dsSave = new DataSet();

			dsSave.Tables.Add(dtShipment);
			
			if(dtBloodUnit != null)
			{
				dsSave.Tables.Add(dtBloodUnit);
				dsSave.Tables.Add(dtBloodUnitFinance);
				dsSave.Tables.Add(dtBloodUnitMedia);
				dsSave.Tables.Add(dtBloodUnitStatus);
			}
			if(dtWorkloadEvents != null)
			{
				dsSave.Tables.Add(dtWorkloadEvents);
			}
			if(dtBloodUnitAntigens != null)
			{
				dsSave.Tables.Add(dtBloodUnitAntigens);
			}
			if(dtBloodUnitTests != null)
			{
				dsSave.Tables.Add(dtBloodUnitTests);
			}
			if(dtBloodUnitShipment != null)
			{
				dsSave.Tables.Add(dtBloodUnitShipment);
			}
			//BR_1.61
			if(bloodUnit.BloodUnitMedia.Comments.Length > 0)
			{
				//generate exception if expired unit was received (required to save data)
				BOL.ExceptionReport exRep = new BOL.ExceptionReport(Common.ExceptionType.Expired_unit_received, bloodUnit.BloodUnitMedia.Comments, Common.LogonUser.LogonUserName);
				exRep.GenerateExceptionExpiredUnitReceived(bloodUnit);
				
				if (dtExceptionExpiredUnitReceived == null)
				{
					dtExceptionExpiredUnitReceived = exRep.ExceptionData.Table.Clone();
				}
				dtExceptionExpiredUnitReceived.Rows.Add(exRep.ExceptionData.ItemArray);
				dsSave.Tables.Add(dtExceptionExpiredUnitReceived);
			}
			
			DAL.Shipment.SaveIncomingShipment(this.ShipmentGuid, reenteredUnit, dsSave, lastUpdateFunctionId);
        }

		
        

		#endregion

    }
}


